<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\Blog;
use Illuminate\Support\Str;
use App\Models\BlogCategory;
use Illuminate\Http\Request;
use App\Http\Requests\StoreBlog;
use App\Http\Requests\UpdateBlog;
use Yajra\Datatables\Datatables;
use Illuminate\Support\Facades\Storage;


class BlogController extends Controller
{
    public function index()
    {
        if(!auth()->user()->can('view_blog')) {
            abort(403, 'Unauthorized Action');
        }
        return view('dashboards.admins.blog.index');
    }

    public function ssd(Request $request)
    {
        if(!auth()->user()->can('view_blog')) {
            abort(403, 'Unauthorized Action');
        }

        $blog = Blog::with('blogcategories');

        return Datatables::of($blog)
            ->filterColumn('blogcategories', function($query, $keyword) {
                $query->whereHas('blogcategories', function($q1) use ($keyword) {
                    $q1->where('name', 'like', '%'.$keyword.'%');
                });
            })
            ->editColumn('picture', function($each) {
                return '<img src="' . $each->blogimage_path() . '" class="profile-thumbnail" alt=""/><p class="my-1">' . $each->name .'</p>';
            })
            ->editColumn('blogcategories', function (Blog $each) {
                return $each->blogcategories->map(function($categories){
                    return $categories->name;
                })->implode(' / ');
            })
            ->editColumn('description', function($each) {
                return Str::limit($each->description, 300);
            })
            ->editColumn('updated_at', function($each) {
                return Carbon::parse($each->updated_at)->format('Y-m-d H:i:s');
            })
            ->addColumn('action', function($each) {
                $edit_icon = '<a href="' . route('blog.edit', $each->id) . '" class="text-warning"><i class="far fa-edit"></i></a>';
                $info_icon = '<a href="' . route('blog.show', $each->id) . '" class="text-primary"><i class="fas fa-info-circle"></i></a>';
                $delete_icon = '<a href="#" class="text-danger delete-btn" data-id="'. $each->id .'"><i class="fas fa-trash-alt"></i></a>';

                return '<div class="action-icon">' . $edit_icon . $info_icon . $delete_icon.'</div>';
            })
            ->addColumn('plus-icon', function($each) {
                return null;
            })
            ->rawColumns(['description', 'action', 'picture'])
            ->make(true);
    }

    public function create()
    {
        if(!auth()->user()->can('create_blog')) {
            abort(403, 'Unauthorized Action');
        }

        $category = BlogCategory::orderBy('name')->get();
        return view('dashboards.admins.blog.create', compact('category'));
    }

    public function store(StoreBlog $request)
    {
        if(!auth()->user()->can('create_blog')) {
            abort(403, 'Unauthorized Action');
        }

        $picture = null;
        if($request->hasFile('picture')) {
            $picture = $request->file('picture');
            $picture_name = uniqid(). '_' . time() . '.' . $picture->getClientOriginalExtension();
            Storage::disk('public')->put('blog/' . $picture_name, file_get_contents($picture));
        }

        $user_id = auth()->user()->id;

        $slug = uniqid();

        $blog = new Blog();
        $blog->user_id = $user_id;
        $blog->slug = $slug;
        $blog->title = $request->title;
        $blog->description = $request->description;
        $blog->view_count = 0;
        $blog->picture = $picture_name;
        $blog->status = $request->status;
        $blog->save();

        $blog->blogcategories()->sync($request->blogcategories, false);

        return redirect()->route('blog.index')->with('create', 'Blog is successfully Create!');
    }

    public function edit($id)
    {
        if(!auth()->user()->can('edit_blog')) {
            abort(403, 'Unauthorized Action');
        }

        $blog = Blog::findOrFail($id);
        $category = BlogCategory::all();
        $cat = array();
        foreach ($category as $c) {
            $cat[$c->id] = $c->name;
        }

        return view('dashboards.admins.blog.edit')->withBlog($blog)->withCategory($cat);
    }
    public function update($id, UpdateBlog $request)
    {
        if(!auth()->user()->can('edit_blog')) {
            abort(403, 'Unauthorized Action');
        }

        $blog = Blog::findOrFail($id);

        $picture_name = $blog->picture;
        if($request->hasFile('picture')) {

            Storage::disk('public')->delete('blog/' . $blog->picture);

            $picture = $request->file('picture');
            $picture_name = uniqid(). '_' . time() . '.' . $picture->getClientOriginalExtension();
            Storage::disk('public')->put('blog/'. $picture_name, file_get_contents($picture));
        }

        $blog->title = $request->title;
        $blog->description = $request->description;
        $blog->picture = $picture_name;
        $blog->status = $request->status;
        $blog->update();

        if(isset($request->blogcategories)) {
            $blog->blogcategories()->sync($request->blogcategories);
        }else{
            $blog->blogcategories()->sync(array());
        }

        return redirect()->route('blog.index')->with('update', 'Blog is successfully Update!');
    }

    public function show($id)
    {
        if(!auth()->user()->can('view_blog')) {
            abort(403, 'Unauthorized Action');
        }
        $blog = Blog::findOrFail($id);
        return view('dashboards.admins.blog.show', compact('blog'));
    }

    public function destroy($id)
    {
        if(!auth()->user()->can('delete_blog')) {
            abort(403, 'Unauthorized Action');
        }
        $blog = Blog::findOrFail($id);
        $blog->delete();
        Storage::disk('public')->delete('blog/' . $blog->picture);
        return 'success';
    }
}
