<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\Department;
use App\Models\User;
use Illuminate\Http\Request;
use Yajra\Datatables\Datatables;
use Spatie\Permission\Models\Role;
use App\Http\Requests\StoreEmployee;
use Illuminate\Support\Facades\Hash;
use App\Http\Requests\UpdateEmployee;
use Illuminate\Support\Facades\Storage;

class EmployeeController extends Controller
{
    public function index()
    {
        if(!auth()->user()->can('view_user')) {
            abort(403, 'Unauthorized Action');
        }
        return view('dashboards.admins.employee.index');
    }

    public function ssd(Request $request)
    {
        if(!auth()->user()->can('view_user')) {
            abort(403, 'Unauthorized Action');
        }

        $employee = User::with('department');

        return Datatables::of($employee)
            ->filterColumn('department_name', function($query, $keyword) {
                $query->whereHas('department', function($q1) use ($keyword) {
                    $q1->where('title', 'like', '%'.$keyword.'%');
                });
            })
            ->editColumn('profile_img', function($each) {
                return '<img src="' . $each->profile_img_path() . '" class="profile-thumbnail" alt=""/><p class="my-1">' . $each->name .'</p>';
            })
            ->addColumn('department_name', function($each){
                return $each->department ? $each->department->title : '-';
            })
            ->addColumn('role_name', function($each) {
                $output = '';
                foreach($each->roles as $role){
                    $output .= '<span class="badge badge-pill badge-primary">'. $role->name .'</span>';
                }
                return $output;
            })
            ->editColumn('is_present', function($each) {
                if($each->is_present == 1){
                    return '<span class="badge badge-pill badge-success">Present</span>';
                }else{
                    return '<span class="badge badge-pill badge-danger">Leave</span>';
                }
            })
            ->editColumn('updated_at', function($each) {
                return Carbon::parse($each->updated_at)->format('Y-m-d H:i:s');
            })
            ->addColumn('action', function($each) {
                $edit_icon = '<a href="' . route('employee.edit', $each->id) . '" class="text-warning"><i class="far fa-edit"></i></a>';
                $info_icon = '<a href="' . route('employee.show', $each->id) . '" class="text-primary"><i class="fas fa-info-circle"></i></a>';
                $delete_icon = '<a href="#" class="text-danger delete-btn" data-id="'. $each->id .'"><i class="fas fa-trash-alt"></i></a>';

                return '<div class="action-icon">' . $edit_icon . $info_icon . $delete_icon.'</div>';
            })
            ->addColumn('plus-icon', function($each) {
                return null;
            })
            ->rawColumns(['role_name', 'is_present', 'action', 'profile_img'])
            ->make(true);
    }

    public function create()
    {
        if(!auth()->user()->can('create_user')) {
            abort(403, 'Unauthorized Action');
        }

        $departments = Department::orderBy('title')->get();
        $roles = Role::all();
        return view('dashboards.admins.employee.create', compact('departments', 'roles'));
    }

    public function store(StoreEmployee $request)
    {
        if(!auth()->user()->can('create_user')) {
            abort(403, 'Unauthorized Action');
        }

        $profile_img_name = null;
        if($request->hasFile('profile_img')) {
            $profile_img_file = $request->file('profile_img');
            $profile_img_name = uniqid(). '_' . time() . '.' . $profile_img_file->getClientOriginalExtension();
            Storage::disk('public')->put('employee/' . $profile_img_name, file_get_contents($profile_img_file));
        }

        $employee = new User();
        $employee->employee_id = $request->employee_id;
        $employee->name = $request->name;
        $employee->phone = $request->phone;
        $employee->email = $request->email;
        $employee->nrc_number = $request->nrc_number;
        $employee->gender = $request->gender;
        $employee->birthday = $request->birthday;
        $employee->address = $request->address;
        $employee->department_id = $request->department_id;
        $employee->date_of_join = $request->date_of_join;
        $employee->is_present = $request->is_present;
        $employee->profile_img = $profile_img_name;
        $employee->password = Hash::make($request->password);
        $employee->save();

        $employee->syncRoles($request->roles);

        return redirect()->route('employee.index')->with('create', 'User is successfully Create!');
    }

    public function edit($id)
    {
        if(!auth()->user()->can('edit_user')) {
            abort(403, 'Unauthorized Action');
        }

        $employee = User::findOrFail($id);
        $old_roles = $employee->roles->pluck('id')->toArray();
        $departments = Department::orderBy('title')->get();
        $roles = Role::all();

        return view('dashboards.admins.employee.edit', compact('employee', 'old_roles', 'departments', 'roles'));
    }
    public function update($id, UpdateEmployee $request)
    {
        if(!auth()->user()->can('edit_user')) {
            abort(403, 'Unauthorized Action');
        }

        $employee = User::findOrFail($id);

        $profile_img_name = $employee->profile_img;
        if($request->hasFile('profile_img')) {

            Storage::disk('public')->delete('employee/' . $employee->profile_img);

            $profile_img_file = $request->file('profile_img');
            $profile_img_name = uniqid(). '_' . time() . '.' . $profile_img_file->getClientOriginalExtension();
            Storage::disk('public')->put('employee/'. $profile_img_name, file_get_contents($profile_img_file));
        }

        $employee->employee_id = $request->employee_id;
        $employee->name = $request->name;
        $employee->phone = $request->phone;
        $employee->email = $request->email;
        $employee->nrc_number = $request->nrc_number;
        $employee->gender = $request->gender;
        $employee->birthday = $request->birthday;
        $employee->address = $request->address;
        $employee->department_id = $request->department_id;
        $employee->date_of_join = $request->date_of_join;
        $employee->is_present = $request->is_present;
        $employee->profile_img = $profile_img_name;
        $employee->password = $request->password ? Hash::make($request->password) : $employee->password;
        $employee->update();

        $employee->syncRoles($request->roles);

        return redirect()->route('employee.index')->with('update', 'User is successfully Update!');
    }

    public function show($id)
    {
        if(!auth()->user()->can('view_user')) {
            abort(403, 'Unauthorized Action');
        }
        $employee = User::findOrFail($id);
        return view('dashboards.admins.employee.show', compact('employee'));
    }

    public function destroy($id)
    {
        if(!auth()->user()->can('delete_user')) {
            abort(403, 'Unauthorized Action');
        }
        $employee = User::findOrFail($id);
        $employee->delete();

        return 'success';
    }
}
